! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! *
! * SUBROUTINE:          TYPE 301 - Temperature Sensor
! *
! * PURPOSE:             Calculate the response of a first order linear
! *                      temperature sensor
! *
! * MAJOR RESTRICTIONS:  First order, linear
! *
! * DEVELOPER:           Li Mei and Philip Haves
! *                      Loughborough University
! *
! * LAST MODIFIED:       November 22, 1995
! *
! * INCLUDE FILES:       None
! * SUBROUTINES CALLED:  None
! * FUNCTIONS  CALLED:   None
! *
! * REFERENCE:           Based on HVACSIM+ Type7 (NIST)
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. ti      : temperature measured                               (C)
! *
! * OUTPUTS
! * =======
! *  1. co      : sensor output                                      (-)
! *
! * PARAMETERS
! * ==========
! *  1. tzero   : offset: input for zero output                      (C)
! *  2. tgain   : gain: change in input for 0->1 output              (K)
! *  3. tcon    : time constant                                      (s)
! *  4. cmax    : upper limit of output range                        (-)
! *  5. cmin    : lower limit of output range                        (-)
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call
! *  2. ts      : solution of differential equation from previous call
! *  3. tsp     : solution of differential equation from previous step
! *               time
! *
! * INTERNAL VARIABLES
! * ==================
! * (None)
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type301(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=1,no=1,np=5,&
                                             ndiffeq=1,ns=1+ndiffeq*2
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: ti,tgain,tcon,cmax,cmin,tsp,ts,aa,bb,tsba,ci,&
                        co,tsbar

! **** Read in input
        ti = xin(1)
! **** Read in parameters
        tzero = par_v(1)
        tgain = par_v(2)
        if (tgain==0.0) then
            stop 'type 301: temperature range must be non-zero'
        endif
        tcon = par_v(3)
        cmax = par_v(4)
        cmin = par_v(5)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.0
            endif
            if (init==0) then
! **** Sensor is in equilibrium with environment at start of simulation
                saved_v(2) = ti
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update value of sensor temperature from
! **** previous time-step
            saved_v(3) = saved_v(2)
        endif
! **** Update previous values
        tsp = saved_v(3)
! **** Determine temperature of sensor element
        if (tcon<=0.0) then
! **** Instantaneous response
            ts = ti
        else
! **** Non-zero time constant - integrate analytically
            aa = -1./tcon
            bb = ti/tcon
            call diffeq(time,aa,bb,tsp,ts,tsbar)
        endif
! **** Determine sensor output
! **** Calculate unclipped sensor output from range and offset
        ci = (ts - tzero)/tgain
! **** limit output to user-specified range
        if (ci>cmax) then
            co = cmax
        elseif (ci<cmin) then
            co = cmin
        else
            co = ci
        endif
! **** Save provisional value to be used in updating at next step time
        saved_v(2) = ts
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = co
! **** Determine whether to allow freezing
! **** Freezing of the output is allowed only if the input is a constant
! **** or a boundary variable and the change in the output is small
        if (tcon==0.0 .or. ((iostat(1)<-1).or.(iostat(1)==2))&
              .and. ((abs(ti - ts))<=(rtolx * abs(ts)+atolx))) then
            iostat(1) = 1
        else
            iostat(1) = 0
        endif
! **** Return
        return
        end subroutine type301
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! *
! * SUBROUTINE:          TYPE 311 - Temperature Sensor
! *
! * PURPOSE:             Calculate the response of a first order linear
! *                      temperature sensor
! *
! * MAJOR RESTRICTIONS:  First order, linear
! *
! * DEVELOPER:           Li Mei and Philip Haves
! *                      Loughborough University
! *
! * LAST MODIFIED:       December 12, 2012 by Drexel
! *
! * INCLUDE FILES:       None
! * SUBROUTINES CALLED:  None
! * FUNCTIONS  CALLED:   None
! *
! * REFERENCE:           Based on HVACSIM+ Type7 (NIST)
! *
! **********************************************************************
! * INPUTS
! * ======
! *  1. ti      : temperature measured                               (C)
! *
! * OUTPUTS
! * =======
! *  1. co      : sensor output                                      (-)
! *
! * PARAMETERS
! * ==========
! *  1. tzero   : offset: input for zero output                      (C)
! *  2. tgain   : gain: change in input for 0->1 output              (K)
! *  3. tcon    : time constant                                      (s)
! *  4. cmax    : upper limit of output range                        (-)
! *  5. cmin    : lower limit of output range                        (-)
! *  6. location: 1-RA, 2-MA, 3-OA, 4-SA hot deck, 5-SA cold deck    (-)
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call
! *  2. ts      : solution of differential equation from previous call
! *  3. tsp     : solution of differential equation from previous step
! *               time
! *
! * INTERNAL VARIABLES
! * ==================
! * (None)
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type311(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=1,no=1,np=6,&
                                             ndiffeq=1,ns=1+ndiffeq*2
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: ti,tgain,tcon,cmax,cmin,tsp,ts,aa,bb,tsba,ci,&
                        co,tsbar,location

! **** Read in input
        ti = xin(1)
! **** Read in parameters
        tzero = par_v(1)
        tgain = par_v(2)
        if (tgain==0.0) then
            stop 'type 311: temperature range must be non-zero'
        endif
        tcon     = par_v(3)
        cmax     = par_v(4)
        cmin     = par_v(5)
        location = nint(par_v(6))
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.0
            endif
            if (init==0) then
! **** Sensor is in equilibrium with environment at start of simulation
                saved_v(2) = ti
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update value of sensor temperature from
! **** previous time-step
            saved_v(3) = saved_v(2)
        endif
! **** Update previous values
        tsp = saved_v(3)
! **** Determine temperature of sensor element
        if (tcon<=0.0) then
! **** Instantaneous response
            ts = ti
        else
! **** Non-zero time constant - integrate analytically
            aa = -1./tcon
            bb = ti/tcon
            call diffeq(time,aa,bb,tsp,ts,tsbar)
        endif
! **** Determine sensor output
! **** Calculate unclipped sensor output from range and offset
        ci = (ts - tzero)/tgain
! **** limit output to user-specified range
        if (ci>cmax) then
            co = cmax
        elseif (ci<cmin) then
            co = cmin
        else
            co = ci
        endif
! **** Save provisional value to be used in updating at next step time
        saved_v(2) = ts
! **** Save time of current call
        saved_v(1) = time

! ---- Fault flag on
	
		if (location==1)then
			if (RAtemp==0) then
				co = co
			elseif (RAtemp==1) then
				co = co + VRAtemp
			endif
        elseif (location==2)then
			if (MAtemp==0) then
				co = co
			elseif (MAtemp==1) then
				co = co + VMAtemp
			endif
        elseif (location==3)then
			if (OAtemp==0) then
				co = co
			elseif (OAtemp==1) then
				co = co + VOAtemp
			endif
        elseif (location==4)then
			if (SAhottemp==0) then
				co = co
			elseif (SAhottemp==1) then
				co = co + VSAhottemp
			endif
        elseif (location==5)then
			if (SAcoldtemp==0) then
				co = co
			elseif (SAcoldtemp==1) then
				co = co + VSAcoldtemp
			endif
        endif

! ---- Fault flag off

! **** Output
        yout(1) = co
! **** Determine whether to allow freezing
! **** Freezing of the output is allowed only if the input is a constant
! **** or a boundary variable and the change in the output is small
        if (tcon==0.0 .or. ((iostat(1)<-1).or.(iostat(1)==2))&
              .and. ((abs(ti - ts))<=(rtolx * abs(ts)+atolx))) then
            iostat(1) = 1
        else
            iostat(1) = 0
        endif
! **** Return
        return
        end subroutine type311

! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! *
! * SUBROUTINE:          TYPE 302 - Humidity sensor
! *
! * PURPOSE:             Calculate the response of a first order linear
! *                      humidity sensor
! *
! * MAJOR RESTRICTIONS:  First order, linear
! *
! * DEVELOPER:           Li Mei and Philip Haves
! *                      Loughborough University of Technology
! *
! * LAST MODIFIED:       November 22, 1995
! *
! * INCLUDE FILES:       None
! * SUBROUTINES CALLED:  None
! * FUNCTIONS  CALLED:   FPHI,FTDEW,FTWB
! *
! * REFERENCE:
!
! **********************************************************************
! * INPUTS
! * ======
! *  1. gi      : humidity ratio                                 (kg/kg)
! *  2. ti      : temperature                                        (C)
! *
! * OUTPUTS
! * =======
! *
! *  2. co      : sensor output                                      (-)
! *
! * PARAMETERS
! * ==========
! *  1. mode    : humidity ratio=1, relative humidity=2, dew point=3,
! *               wet bulb=4, deg of saturation=5, specific enthalpy=6
! *  2. xzero   : offset: input for zero output        (sensed quantity)
! *  3. xgain   : gain: change in i/p for 0->1 o/p     (sensed quantity)
! *  4. tcon    : time constant                                      (s)
! *  5. cmax    : upper limit of output range                        (-)
! *  6. cmin    : lower limit of output range                        (-)
! *
! *  SAVED
! *  =====
! *  1. time    : time of previous call
! *  2. hs      : solution of differential equation from previous call
! *  3. hsp     : solution of differential equation from previous
! *               step-time
! *
! * INTERNAL VARIABLES
! * ==================
! * (None)
! *
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type302(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=2,no=1,np=6,&
                                             ndiffeq=1,ns=1+ndiffeq*2
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

! **** Thermophysical constants
        real         :: patmpa=101325.0
        real         :: gi,ti,xzero,xgain,tcon,cmax,cmin,xi,fphi,ftdew,&
                        ftwb,gs,fwphi,fhair,xsp,xs,aa,bb,xsbar,ci,co
        integer      :: mode

! **** Read in input
        gi      = xin(1)
        ti      = xin(2)
! **** Read in parameters
        mode    = nint(par_v(1))
        xzero   = par_v(2)
        xgain   = par_v(3)
        if (xgain==0.0) then
            stop 'type 302: humidity range must be non-zero'
        endif
        tcon    = par_v(4)
        cmax    = par_v(5)
        cmin    = par_v(6)
! **** Calculate required measure of humidity
        select case (mode)
            case (1)
! **** Humidity ratio
               xi = gi
            case (2)
! **** Relative humidity
               xi = fphi(ti,gi,patmpa)
            case (3)
! **** Dew point temperature
               xi = ftdew(gi,patmpa)
            case (4)
! **** Wet bulb temperature
               xi = ftwb(ti,gi,patmpa)
            case (5)
! **** Degree of saturation
               gs  = fwphi(ti,100.,patmpa)
               xi = gi/gs
            case (6)
! **** Specific enthalpy
               xi = fhair(ti,gi)
            case default
        end select

! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
! **** Sensor is in equilibrium with environment at start of simulation
                saved_v(2) = xi
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update value of sensor temperature from
! **** previous time-step
            saved_v(3) = saved_v(2)
        endif
! **** Update previous values
        xsp = saved_v(3)
! **** Determine humidity of sensor element
        if (tcon<=0.0) then
! **** Instantaneous response
            xs = xi
        else
! **** Non-zero time constant - integrate analytically
            aa = -1./tcon
            bb = xi/tcon
            call diffeq(time,aa,bb,xsp,xs,xsbar)
        endif
! **** Determine sensor output
! **** Calculate unclipped sensor output from range and offset
        ci = (xs - xzero)/xgain
! **** Limit output to user-specified range
        if (ci>cmax) then
            co = cmax
        elseif (ci<cmin) then
            co = cmin
        else
            co = ci
        endif
! **** Save provisional value to be used in updating at next step time
        saved_v(2) = xs
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = co
! **** Determine whether to allow freezing
! **** Freezing of the output is allowed only if the input is a constant
! **** or a boundary variable and the change in the output is small
        if (tcon==0.0 .or. ((iostat(1)<-1).or.(iostat(1)==2))&
!            .and. ((iostat(2)<-1).or.(iostat(2)==2)) .and.& 
		!MAG comment out iostat(2) code, add .and. below
            .and.((abs(xi - xs))<=(rtolx * abs(xs)+atolx))) then
            iostat(1) = 1
        else
            iostat(1) = 0
        endif
! **** Return
        return
        end subroutine type302
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:   Flow rate sensor
! *
! * PURPOSE:      Calculate the response of a first order linear
! *               flow rate sensor
! **********************************************************************
! * INPUTS
! * ======
! *  1. wi      : mass flow rate                                  (kg/s)
! *
! * OUTPUTS
! * =======
! *  1. co      : sensor output                                      (-)
! *
! * PARAMETERS
! * ==========
! *  1. mode    : mass flow rate = 1, volumetric flow rate = 2
! *               velocity = 3, velocity pressure = 4                (-)
! *  2. ifluid  : air = 1, water = 2                                 (-)
! *  3. area    : cross-sectional area of duct or pipe              (m2)
! *  4. xzero   : offset: input when output is zero    (sensed quantity)
! *  5. xgain   : gain: input change for output 0->1   (sensed quantity)
! *  6. tcon    : time constant                                      (s)
! *  7. cmax    : upper limit of output range                        (-)
! *  8. cmin    : lower limit of output range                        (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call                              (s)
! *  2. xsp     : solution of differential equation from previous call
! *  3. xsp     : solution of differential equation from previous step
! *               time
! *
! **********************************************************************
!
!   MAJOR RESTRICTIONS:  First order, linear
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 22, 1994
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  None
!   FUNCTIONS  CALLED:   None
!
!   REVISION HISTORY:    None
!
!   REFERENCE:
!
! **********************************************************************
!
!   INTERNAL VARIABLES:  None
!
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type303(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=1,no=1,np=8,&
                                             ndiffeq=1,ns=1+ndiffeq*2
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: rhoa=1.2,rhow=1000.
        real         :: wi,area,xzero,xgain,tcon,cmax,cmin,rho,xi,xsp,&
                        xs,aa,bb,xsbar,ci,co
        integer      :: mode,ifluid

! **** Read in input
        wi = xin(1)
! **** Read in parameters
        mode    = nint(par_v(1))
        ifluid  = nint(par_v(2))
        area    = par_v(3)
        xzero   = par_v(4)
        xgain   = par_v(5)
        if (xgain==0.0) then
            stop 'type 303: mass flow rate range must be non-zero'
        endif
        tcon    = par_v(6)
        cmax    = par_v(7)
        cmin    = par_v(8)
! **** Determine different modes for mass flow sensor
        if (ifluid==1) then
            rho = rhoa
        else
            rho = rhow
        endif
        if (area<=0.0) stop 'type303: area must be > 0'

        select case (mode)
            case (1)
               xi = wi

            case (2)
               xi = wi/rho
            case (3)
               xi = wi/(area*rho)
            case (4)
               xi = 0.5*wi/(rho*area*area)
            case default
        end select

! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
! **** Sensor is in equilibrium with environment at start of simulation
                saved_v(2) = xi
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update value of sensor temperature from
! **** previous time-step
            saved_v(3) = saved_v(2)
        endif
! **** Update previous values
        xsp = saved_v(3)
! **** Determine state of sensor element
        if (tcon<=0.0) then
! **** Instantaneous response
            xs = xi
        else
! **** Non-zero time constant - integrate analytically
            aa = -1./tcon
            bb = xi/tcon
            call diffeq(time,aa,bb,xsp,xs,xsbar)
        endif
! **** Determine sensor output
! **** Calculate unclipped sensor output from range and offset
        ci = (xs - xzero)/xgain
! **** Limit output to user-specified range
        if (ci>cmax) then
            co = cmax
        elseif (ci<cmin) then
            co = cmin
        else
            co = ci
        endif
! **** Save provisional value to be used in updating at next step time
        saved_v(2) = xs
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = co
! **** Determine whether to allow freezing
! **** Freezing of the output is allowed only if the input is a constant
! **** or a boundary variable and the change in the output is small
        if (tcon==0.0 .or. ((iostat(1)<-1).or.(iostat(1)==2))&
              .and. ((abs(xi-xs))<=(rtolx * abs(xs)+atolx))) then
            iostat(1) = 1
        else
            iostat(1) = 0
        endif
! **** Return
        return
        end subroutine type303
! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Total pressure sensor
! *
! * PURPOSE:     Calculate the response of a first order linear
! *              total pressure sensor
! **********************************************************************
! * INPUTS
! * ======
! *  1. pi      : pressure input                                   (kPa)
! *
! * OUTPUTS
! * =======
! *  1. co      : sensor output                                      (-)
! *
! * PARAMETERS
! * ==========
! *  1. pzero   : offset: input for zero output                    (kPa)
! *  2. pgain   : gain: change in input for 0->1 output            (kPa)
! *  3. tcon    : time constant                                      (s)
! *  4. cmax    : upper limit of output range                        (-)
! *  5. cmin    : lower limit of output range                        (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. ps      : solution of differential equation from previous call
! *  3. psp     : solution of differential equation from previous step
! *               time
! *
! ***********************************************************************
!
!   MAJOR RESTRICTIONS:  First order, linear
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 29, 1994
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  None
!   FUNCTIONS  CALLED:   None
!
!   REVISION HISTORY:    None
!
!   REFERENCE:
!
! **********************************************************************
!
!   INTERNAL VARIABLES:  None
!
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type304(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=1,no=1,np=5,&
                                             ndiffeq=1,ns=1+ndiffeq*2
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: pi,pzero,pgain,tcon,cmax,cmin,psp,ps,aa,bb,&
                        psbar,ci,co

! **** Read in input
        pi    = xin(1)
! **** Read in parameters
        pzero = par_v(1)
        pgain = par_v(2)
        if (pgain==0.0) then
            stop 'type 304: pressure range must be non-zero'
        endif
        tcon  = par_v(3)
        cmax  = par_v(4)
        cmin  = par_v(5)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                 saved_v(1) = -999999.
            endif
            if (init==0) then
! **** Sensor is in equilibrium with environment at start of simulation
                saved_v(2) = pi
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update value of sensor temperature from
! **** previous time-step
            saved_v(3) = saved_v(2)
        endif
! **** Update previous values
        psp = saved_v(3)
! **** Determine state of sensor element
        if (tcon<=0.0) then
! **** Instantaneous response
            ps = pi
        else
! **** Non-zero time constant - integrate analytically
            aa = -1./tcon
            bb = pi/tcon
            call diffeq(time,aa,bb,psp,ps,psbar)
        endif
! **** Determine sensor output
! **** Calculate unclipped sensor output from range and offset
        ci = (ps - pzero)/pgain
! **** Limit output to user-specified range
        if (ci>cmax) then
            co = cmax
        elseif (ci<cmin) then
            co = cmin
        else
            co = ci
        endif
! **** Save provisional value to be used in updating at next step time
        saved_v(2) = ps
! **** Save time of current call
        saved_v(1) = time
! **** Output
        yout(1) = co
! **** Determine whether to allow freezing
! **** Freezing of the output is allowed only if the input is a constant
! **** or a boundary variable and the change in the output is small
        if (tcon==0.0 .or. ((iostat(1)<-1).or.(iostat(1)==2))&
               .and. ((abs(pi - ps))<=(rtolx * abs(ps)+atolx))) then
            iostat(1) = 1
        else
            iostat(1) = 0
        endif
! **** Return
        return
        end subroutine type304


! **********************************************************************
! * Copyright ASHRAE.   Control Simulation Testbed
! **********************************************************************
! * SUBROUTINE:  Static pressure sensor
! *
! * PURPOSE:     Calculate the response of a first order linear
! *              static pressure sensor
! *********************************************************************
! * INPUTS
! * ======
! *  1. ptot    : pressure input (total)                          (kPa)
! *  2. wi      : mass flow rate                                 (kg/s)
! *
! * OUTPUTS
! * =======
! *  1. co      : sensor output                                     (-)
! *
! * PARAMETERS
! * ==========
! *  1. ifluid  : air = 1, water = 2                                (-)
! *  2. area    : cross-sectional area                             (m2)
! *  3. pzero   : offset: input for zero output       (sensed quantity)
! *  4. pgain   : gain: change in input for 0->1 output (sensed quantity)
! *  5. tcon    : time constant                                     (s)
! *  6. cmax    : upper limit of output range                       (-)
! *  7. cmin    : lower limit of output range                       (-)
! *  8. location: 1-SA hot deck, 2-SA cold deck                     (-)
! *
! * SAVED
! * =====
! *  1. time    : time of previous call
! *  2. ps      : solution of differential equation from previous call
! *  3. psp     : solution of differential equation from previous step
! *               time
! ***********************************************************************
!
!   MAJOR RESTRICTIONS:  First order, linear
!
!   DEVELOPER:           Li Mei and Philip Haves
!                        Loughborough University of Technology
!
!   DATE:                November 29, 1994
!
!   INCLUDE FILES:       None
!   SUBROUTINES CALLED:  None
!   FUNCTIONS  CALLED:   None
!
!   REVISION HISTORY:    None
!
!   REFERENCE:
!
! **********************************************************************
!
!   Updated to Fortran 90 March 2, 2007 Cheol Park, NIST
!
! **********************************************************************

        subroutine type305(xin,yout,par_v,saved_v,iostat)

        use modsim_head
        implicit none
        integer,parameter                 :: ni=2,no=1,np=8,&
                                             ndiffeq=1,ns=1+ndiffeq*2
        real,dimension(ni),intent(in)     :: xin                      
        real,dimension(no),intent(out)    :: yout                     
        real,dimension(np),intent(in)     :: par_v                    
        real,dimension(ns),intent(in out) :: saved_v                  
        integer,dimension(no),intent(out) :: iostat

        real         :: rhoa=1.2, rhow=1000.
        real         :: ptot,wi,area,pzero,pgain,tcon,cmax,cmin,&
                        rho,pi,psp,ps,aa,bb,psbar,ci,co
        integer      :: ifluid,location

! **** Read in input
        ptot    = xin(1)
        wi      = xin(2)
! **** Read in parameters
        ifluid  = nint(par_v(1))
        area    = par_v(2)
        pzero   = par_v(3)
        pgain   = par_v(4)
        tcon    = par_v(5)
        cmax    = par_v(6)
        cmin    = par_v(7)
		location= nint(par_v(8))
! **** Check air or water flow
        if (ifluid==1) then
            rho = rhoa
        elseif (ifluid==2) then
            rho = rhow
        else
            stop 'type 305: ifluid must be 1 or 2'
        endif
! *****Calculate static pressure
        pi = ptot - 0.001*0.5*wi*wi/(rho*area*area)
! **** Initialize at beginning of simulation
        if (itime<=1) then
            if (init==0 .or. saved_v(1)>time) then
                saved_v(1) = -999999.
            endif
            if (init==0) then
! **** Sensor is in equilibrium with environment at start of simulation
                saved_v(2) = pi
            endif
        endif
        if (time>saved_v(1)) then
! **** First call of timestep - update value of sensor temperature from
! **** previous time-step
            saved_v(3) = saved_v(2)
        endif
! **** Update previous values
        psp = saved_v(3)
! **** Determine state of sensor element
        if (tcon<=0.0) then
! **** Instantaneous response
            ps = pi
        else
! **** Non-zero time constant - integrate analytically
            aa = -1./tcon
            bb = pi/tcon
        call diffeq(time,aa,bb,psp,ps,psbar)
        endif
! **** Determine sensor output
! **** Calculate unclipped sensor output from range and offset
        ci = (ps - pzero)/pgain
! **** Limit output to user-specified range
        if (ci>cmax) then
            co = cmax
        elseif (ci<cmin) then
            co = cmin
        else
            co = ci
        endif
! **** Save provisional value to be used in updating at next step time
        saved_v(2) = ps
! **** Save time of current call
        saved_v(1) = time
! ---- Fault flag on
	
		if (location==1)then
			if (SAhotpress==0) then
				co = co
			elseif (SAhotpress==1) then
				co = co + VSAhotpress
			endif
        elseif (location==2)then
			if (SAcoldpress==0) then
				co = co
			elseif (SAcoldpress==1) then
				co = co + VSAcoldpress
			endif
        endif

! ---- Fault flag off

! **** Output
        yout(1) = co
! **** Determine whether to allow freezing
! **** Freezing of the output is allowed only if the input is a constant
! **** or a boundary variable and the change in the output is small
        if (tcon==0.0 .or. ((iostat(1)<-1).or.(iostat(1)==2))&
            !.and. ((iostat(2)<-1).or.(iostat(2)==2)) .and.&
		! MAG comment out iostat(2) code, add .and. below
            .and.((abs(pi-ps))<=(rtolx * abs(ps)+atolx))) then
            iostat(1) = 1
        else
            iostat(1) = 0
        endif
! **** Return
        return
        end subroutine type305
        
